/**
  @file             SM_global.h
  @version          0.1.0.0

  @brief            
  @details          

  Project           Stepper Motor Driver
  Platform          MPC5606S

  SWVersion         0.1.0


  (c) Copyright 2011 Freescale Semiconductor Inc.
  All Rights Reserved.
*/
/*==================================================================================================
==================================================================================================*/

#ifndef SM_GLOBAL
    #define SM_GLOBAL
	
/*==================================================================================================
                                       DEFINES AND MACROS
==================================================================================================*/

#define SM_TIMER_RES 				8u 							/* 8 MHz timer clock */

#define SM_TIME_10US      			(10u * SM_TIMER_RES)		/* Timer time 10 us */
#define SM_TIME_50US      			(50u * SM_TIMER_RES)		/* Timer time 50 us */
#define SM_TIME_100US      			(100u * SM_TIMER_RES)		/* Timer time 100 us */
#define SM_TIME_500US	   	 		(500u * SM_TIMER_RES)		/* Timer time 500 us */
#define SM_TIME_1MS  	   			(1000u * SM_TIMER_RES)		/* Timer time 1 ms */
#define SM_TIME_1MS1  	   			(1100u * SM_TIMER_RES)		/* Timer time 1,1 ms */
#define SM_TIME_2MS    				(2000u * SM_TIMER_RES)		/* Timer time 2 ms */
#define SM_TIME_2MS2  	   			(2200u * SM_TIMER_RES)		/* Timer time 2,2 ms */	
#define SM_TIME_5MS    				(5000u * SM_TIMER_RES)		/* Timer time 5 ms */
#define SM_TIME_10MS  	 		    (10000u * SM_TIMER_RES)		/* Timer time 10 ms */
#define SM_TIME_20MS  	  		   	(20000u * SM_TIMER_RES)		/* Timer time 20 ms */
#define SM_TIME_50MS  	  		   	(50000u * SM_TIMER_RES)		/* Timer time 50 ms */
#define SM_TIME_100MS  				(100000u * SM_TIMER_RES)	/* Timer time 100 ms */
#define SM_TIME_1S		   			(1000000u * SM_TIMER_RES)	/* Timer time 1 s */

			

#define SM_TABLE_128				/* Sin/Cos table length - number of microsteps in 4 full steps */

#define SM_2_PI						65536u	/* Constant for scaling required position, velocity... */

#define SM_MAX_ACC_MAX				65536u			/* Maximum acceleration limit */ 	/* Need to be qualified */			
#define SM_MAX_ACC_MIN				1000u			/* Maximum acceleration limit */	/* Need to be qualified */
#define SM_MAX_VEL_MAX				1000000u		/* Maximum velocity limit */		/* Need to be qualified */
#define SM_MAX_VEL_MIN				1000u			/* Maximum velocity limit */ 		/* Need to be qualified */
		
#define SM_MTR_FREEZE_TIMEOUT 		SM_TIME_1MS		/* Algorithm timeout during the freeze mode */
#define SM_MTR_RTZ_TIMEOUT			SM_TIME_1MS		/* Algorithm timeout during the RTZ */
#define SM_MTR_RTZ_DONE_TIMEOUT		SM_TIME_1MS		/* Algorithm timeout when RTZ done */
#define SM_MTR_DISABLED_TIMEOUT 	SM_TIME_1MS		/* Algorithm timeout when the motor is disabled */
#define SM_MTR_STOP_TIMEOUT 		SM_TIME_1MS		/* Algodithm timeout when the motor is stopped */

/*==================================================================================================
                                             ENUMS
==================================================================================================*/

typedef enum SM_MotorId				/* Enumerate the motor IDs */ 
{
	SM_MOTOR_ID_1,					/* Motor 1 ID */
	SM_MOTOR_ID_2,					/* Motor 2 ID */
	SM_MOTOR_ID_3,  				/* Motor 3 ID */
	SM_MOTOR_ID_4,					/* Motor 4 ID */
	SM_MOTOR_ID_5,					/* Motor 5 ID */
	SM_MOTOR_ID_6,					/* Motor 6 ID */ 
	SM_MAX_MOTORS					/* Maximum motors number */ 
}SM_MotorId_t;

typedef enum SM_MtrModel
{
	SM_MW2000_FRONT_BASIC_CLASSIC,
	SM_MW2000_FRONT_SILENT_CLASSIC,
	SM_MW2000_REAR_BASIC_CLASSIC,
	SM_SM23,
	SM_MW2000_FRONT_SILENT_FULL_VAL,
	SM_MW2000_FRONT_BASIC_VAL,
	SM_MW2000_FRONT_BASIC_FULL_VAL,
	SM_MW2000_FRONT_NOISE_RED_CLASSIC,
	SM_MW2000_FRONT_NOISE_RED_FULL_VAL,
	SM_MW2000_FRONT_FAIR,
	SM_MW2000_REAR_BASIC_FULL_VAL,
	SM_MW2000_REAR_FAIR
}SM_MtrModel_t;

typedef enum SM_NominalDir
{
	SM_NOM_CLOCKWISE,
	SM_NOM_COUNTER_CLOCKWISE
}SM_NominalDir_t;

typedef enum SM_AlgoType
{
	SM_ALGO_VAR_STEP_TIME,
	SM_ALGO_VAR_STEP_SIZE,
}SM_AlgoType_t;

typedef enum SM_UpdateTime
{
	SM_UPDATE_TIME_1MS1 = SM_TIME_1MS1,
	SM_UPDATE_TIME_2MS2 = SM_TIME_2MS2
}SM_UpdateTime_t;

typedef enum SM_RtzType
{
	SM_RTZ_TYPE_1QUAD,
	SM_RTZ_TYPE_4QUAD
}SM_RtzType_t;

typedef enum SM_RtzQuadrant
{
	SM_RTZ_QUAD_0,
	SM_RTZ_QUAD_1,
	SM_RTZ_QUAD_2,
	SM_RTZ_QUAD_3
}SM_RtzQuadrant_t;

typedef enum SM_ActualVariable
{
	SM_ACTUAL_POSITION,
	SM_ACTUAL_SPEED,
	SM_ACTUAL_ACCELERATION,
	SM_ACTUAL_DAMPING_FACTOR,
	SM_ACTUAL_TARGET_POSITION,
	SM_ACTUAL_ACCELERATION_LIMIT,
	SM_ACTUAL_DECELERATION_LIMIT,
	SM_ACTUAL_MAXIMUM_SPEED,
	SM_ACTUAL_THRESHOLD_1,
	SM_ACTUAL_THRESHOLD_2,
	SM_ACTUAL_THRESHOLD_3,
	SM_ACTUAL_THRESHOLD_4
}SM_ActualVariable_t;

typedef enum SM_ZeroDetected
{
	SM_ZERO_NOT_DETECTED,
	SM_ZERO_DETECTED
}SM_ZeroDetected_t;

typedef enum SM_MotorHwInitState	/* Enumerate possible hardware initialization status */ 
{
	SM_HW_MTR_DISABLED,				/* Motor hardware disabled (motor HW modules not initialized yet) */
	SM_HW_MTR_INITIALIZED,			/* Motor hardware initialized */
}SM_MotorHwInitState_t;

typedef enum SM_MotorEnable			/* Enumerate possible motor driver status */ 
{
	SM_MTR_DISABLED,				/* Motor driver disabled (each motor separately) */
	SM_MTR_ENABLED,					/* Motor driver enabled (each motor separately) */
}SM_MotorEnable_t;	

typedef enum SM_MovementDirection	/* Enumerate possible motor direction */
{	
	SM_DIRECTION_CW,				/* Stepper motor direction - clockwise */
	SM_DIRECTION_CCW,				/* Stepper motor direction - counterclockwise */ 
}SM_MovementDirection_t;	

typedef enum SM_FreezeState			/* Enumerate possible freeze state */
{
	SM_FREEZE_STATE_UNFREEZE,		/* Freeze state - not frozen */	
	SM_FREEZE_STATE_FREEZE,			/* Freeze state - frozen */
}SM_FreezeState_t;

typedef enum SM_RtzThreshold
{
	SM_RTZ_THRESHOLD_MV350,
	SM_RTZ_THRESHOLD_MV450,
	SM_RTZ_THRESHOLD_MV550,
	SM_RTZ_THRESHOLD_MV650,
	SM_RTZ_THRESHOLD_MV750,
	SM_RTZ_THRESHOLD_MV850 
}SM_RtzThreshold_t;

typedef enum SM_MotorState			/* Enumerate possible motor state */
{
	SM_MOTOR_STATE_STOP,			/* Motor stopped */	
	SM_MOTOR_STATE_RUN, 			/* Motor runs */
	SM_MOTOR_STATE_RTZ_RUN, 		/* Motor running in RTZ */
	SM_MOTOR_STATE_RTZ_DONE, 		/* Motor just finished RTZ */
}SM_MotorState_t;

typedef enum SM_AlgoState			/* Enumerate possible motor algorithm state */
{
	SM_ALGO_STATE_STOP,				/* Motor currently stopped */ 
	SM_ALGO_STATE_RAMPUP,			/* Motor currently ramping up */
	SM_ALGO_STATE_RAMPUP_DONE,		/* Motor just finished ramp up */
	SM_ALGO_STATE_RAMPDOWN,			/* Motor currently ramping down */
	SM_ALGO_STATE_RAMPDOWN_DONE,	/* Motor just finished ramp down */
	SM_ALGO_STATE_SLOWDOWN,			/* Motor currently slowing down */
	SM_ALGO_STATE_RUN,				/* Motor runs on maximum speed */ 
	SM_ALGO_STATE_FROZEN,			/* Motor frozen */
}SM_AlgoState_t;

typedef enum SM_CmdToAlgo			/* Enumerate possible commands to the motor algorithm */
{
	SM_CMD_TO_ALGO_NONE,			/* No command sent */
	SM_CMD_TO_ALGO_UPDATE,			/* Update algorithm variables according to new required parameters */
	SM_CMD_TO_ALGO_FREEZE			/* Freeze motor */
}SM_CmdToAlgo_t;

typedef enum SM_IntResultStat		/* Enumerate possible integration result status */ 
{
	SM_HLL_INT_RESULT_INVALID,		/* Result of the integration is invalid */
	SM_HLL_INT_RESULT_VALID 		/* Result of the integration is valid */
}SM_IntResultStat_t;

typedef enum SM_Err					/* Enumerate possible error report - used in API as a return value of the function */ 
{
	SM_OK,							/* No error occured */
	SM_ERR_INVALID_MTR_ID,			/* Set motor ID is invalid */
	SM_ERR_MTR_NOT_ENABLED,			/* Motor currently not enabled */ 
	SM_ERR_MTR_NOT_FROZEN,			/* Motor is not frozen */
	SM_ERR_MTR_ALREADY_FROZEN,		/* Motor is already frozen */
	SM_ERR_ZERO_NOT_DETECTED
}SM_Err_t;

/*==================================================================================================
                                 STRUCTURES AND OTHER TYPEDEFS
==================================================================================================*/

/* General motor driver structure */
typedef struct 
{
	SM_MotorId_t			eMotorId;				/* Specifies the motor ID that the structure is for */
	SM_MotorEnable_t		eMotorEnable;			/* Motor currently enabled / disabled */
	SM_MotorHwInitState_t	eMotorHwInitialized;	/* Motor HW initialized / not */
    SM_MotorState_t			eMotorState;			/* Specifies what is the motor currently operating */ 
    SM_NominalDir_t			eMotorNominalDir;
    SM_FreezeState_t		eFreeze;				/* Motor freeze state - frozen / not */
    SM_CmdToAlgo_t			eCmdToAlgo;				/* Specifies command to the motor algorithm */
   	SM_AlgoType_t			eAlgoType;				/* Algorithm type */
  	SM_UpdateTime_t			eUpdateTime;
  	SM_ZeroDetected_t		eZeroDetected;
    uint32_t				u32StartVelTimeout;    	/* Start velocity timeout (calculated from maximum acc) */
    uint32_t   				u32ReqMaxVelTimeout;	/* Required maximum velocity timeout (by user) */
	int32_t					s32ReqPos;				/* Required position (by user) */ 
	uint8_t					u8ClockDelay;
    const SM_Table_t 		*cpTableSet;   
}SM_General_t;

/* Motor algorithm parameters stucture */
typedef struct 
{
	SM_MovementDirection_t	eMtrCmdDir;				/* Motor command direction (direction curently required) */
	SM_AlgoState_t			eAlgoState;				/* Motor algorithm state */
    uint16_t   				u16Step;				/* Actual step */
    uint16_t				u16Ramp;    			/* Rapming index */
    int32_t   				s32Pos;					/* Actual position */
    int32_t   				s32CmdPos;				/* Commanded position */
    uint32_t 				u32AbsPosToDest;		/* Absolute value of the distance between actual position and the destination [step] */ 
    uint32_t   				u32MaxVelTimeout;		/* Maximum velocity timeout */
    uint32_t   				u32ActVelTimeout;		/* Actual velocity timeout */
    int16_t					s16TempCalc1;
    int16_t					s16TargetPos;
    uint16_t				u16DampingFactor;
    int16_t					s16ActSpeed;
    int16_t					s16ActPos;
    int16_t					s16OldPos;
    int16_t					s16ActAcc;
    int16_t					s16PrevSpeed;
    uint8_t					u8TblSpeed;
}SM_MtrAlgo_t;

/* Structure specifies mechanical motor parameters */
typedef struct 
{
    int16_t   				s16MotorMaxVel;			/* Motor maximum velocity (defined by user accorting to the motor type) */
	int16_t					s16MotorMaxAcc;			/* Motor maxumum acceleration (defined by user according to the motor type */
   	int16_t					s16MotorMaxDec;
    uint16_t				u16Threshold1;
    uint16_t				u16Threshold2;
    uint16_t				u16Threshold3;
    uint16_t				u16Threshold4;
}SM_MtrType_t;

/* Structure contains motor RTZ movement parameters */
typedef struct 
{
 	SM_IntResultStat_t		eIntResultStat;			/* Actual integration result status - valid / invalid */
   	SM_RtzQuadrant_t		eRtzStartQuad;
   	SM_RtzQuadrant_t		eRtzStopQuad;
    uint16_t   				u16RtzBlankingTime;		/* RTZ Blanking time [us] - MUST be set carefully */ 
    uint16_t   				u16RtzIntegrationTime; 	/* RTZ Integration time - calculated according to required RTZ velocity */
    int16_t  			   	s16RtzBemfThreshold;	/* RTZ Back EMF Threshold to detect the stall - MUST be set very carefully */
 }SM_RtzParams_t;

/* All structures wrapped into one */ 
typedef struct 
{
	SM_General_t			trGeneral;
	SM_MtrType_t			trMtrType;
	SM_MtrAlgo_t			trMtrAlgo;
	SM_RtzParams_t 			trRtz;
}SM_MotorParams_t;


#endif /* SM_GLOBAL */
